// ============================================
// SCRIPTRA CHROME EXTENSION - PREMIUM VERSION
// ============================================

// Platform limits
const PLATFORMS = {
  twitter: { limit: 280, name: 'Twitter / X' },
  linkedin: { limit: 3000, name: 'LinkedIn' },
  instagram: { limit: 2200, name: 'Instagram' },
  threads: { limit: 500, name: 'Threads' },
  facebook: { limit: 63206, name: 'Facebook' },
  tiktok: { limit: 2200, name: 'TikTok' }
};

// Templates
const TEMPLATES = {
  engagement: [
    { title: "Question Hook", content: "What's the one thing you wish you knew before [topic]?\n\nHere's mine: [your answer]\n\nDrop yours below 👇" },
    { title: "Poll Format", content: "Quick poll:\n\n🅰️ [Option A]\n🅱️ [Option B]\n\nVote with an emoji!" },
    { title: "This or That", content: "Let's settle this:\n\n[Option A] or [Option B]?\n\nI'm team [your choice] because [reason]" },
    { title: "Unpopular Opinion", content: "Unpopular opinion:\n\n[Your controversial take]\n\nChange my mind 👇" },
    { title: "Fill in Blank", content: "Complete this sentence:\n\nThe best [topic] advice I ever received was _______" }
  ],
  promotional: [
    { title: "Launch Announcement", content: "🚀 It's finally here!\n\n[Product/Service name] is now live.\n\n✅ [Benefit 1]\n✅ [Benefit 2]\n✅ [Benefit 3]\n\nLink in bio 🔗" },
    { title: "Limited Offer", content: "⏰ [X hours] left!\n\n[Offer description]\n\nUse code [CODE] for [discount]%\n\nDon't miss out 👉 [link]" },
    { title: "Social Proof", content: "\"[Customer quote]\"\n\n— [Customer name]\n\nJoin [X]+ happy customers 🎉\n\n[CTA]" },
    { title: "Before/After", content: "Before [Product]:\n❌ [Pain point 1]\n❌ [Pain point 2]\n\nAfter [Product]:\n✅ [Benefit 1]\n✅ [Benefit 2]\n\nMake the switch 👉" }
  ],
  educational: [
    { title: "Tips Thread", content: "[X] tips to [achieve goal]:\n\n1️⃣ [Tip 1]\n2️⃣ [Tip 2]\n3️⃣ [Tip 3]\n4️⃣ [Tip 4]\n5️⃣ [Tip 5]\n\nBookmark this 🔖" },
    { title: "Myth Buster", content: "Myth: [Common misconception]\n\nReality: [The truth]\n\nHere's why this matters:\n[Explanation]" },
    { title: "Step by Step", content: "How to [achieve result] in [X] steps:\n\nStep 1: [Action]\nStep 2: [Action]\nStep 3: [Action]\n\nSimple but effective." },
    { title: "Quick Fact", content: "Did you know?\n\n[Interesting statistic or fact]\n\nThis means [implication]\n\nWhat do you think? 🤔" }
  ],
  storytelling: [
    { title: "Journey Story", content: "[X] years ago, I [starting point].\n\nToday, I [where you are now].\n\nHere's what I learned:\n\n[Key lesson]" },
    { title: "Failure to Success", content: "I failed at [something].\n\nNot once. Not twice. [X] times.\n\nBut here's what failure taught me:\n\n[Lesson]" },
    { title: "Day in the Life", content: "5AM: [Activity]\n8AM: [Activity]\n12PM: [Activity]\n6PM: [Activity]\n10PM: [Activity]\n\nA day in my life 📅" },
    { title: "Lessons Learned", content: "[X] lessons from [experience/years]:\n\n1. [Lesson 1]\n2. [Lesson 2]\n3. [Lesson 3]\n\nWhich one resonates most?" }
  ]
};

// Hashtags database
const HASHTAGS = {
  marketing: {
    popular: ['#marketing', '#digitalmarketing', '#socialmedia', '#contentmarketing', '#branding', '#business', '#growth', '#strategy'],
    niche: ['#contentcreator', '#growthhacking', '#brandstrategy', '#socialmediastrategy', '#b2bmarketing', '#marketingtips'],
    trending: ['#ai', '#chatgpt', '#automation', '#personalization', '#ugc', '#influencermarketing']
  },
  tech: {
    popular: ['#tech', '#technology', '#coding', '#programming', '#developer', '#software', '#startup', '#innovation'],
    niche: ['#webdev', '#frontend', '#backend', '#devops', '#saas', '#buildinpublic', '#indiemaker'],
    trending: ['#ai', '#machinelearning', '#chatgpt', '#web3', '#blockchain', '#metaverse']
  },
  business: {
    popular: ['#business', '#entrepreneur', '#success', '#motivation', '#money', '#startup', '#leadership', '#ceo'],
    niche: ['#smallbusiness', '#businesstips', '#entrepreneurlife', '#sidehustle', '#passiveincome', '#freelancer'],
    trending: ['#remotework', '#solopreneur', '#personalbranding', '#linkedintips', '#networking']
  },
  fitness: {
    popular: ['#fitness', '#gym', '#workout', '#health', '#motivation', '#bodybuilding', '#training', '#fit'],
    niche: ['#homeworkout', '#strengthtraining', '#cardio', '#nutrition', '#mealprep', '#fitnessjourney'],
    trending: ['#wellness', '#mentalhealth', '#selfcare', '#mindfulness', '#biohacking']
  },
  lifestyle: {
    popular: ['#lifestyle', '#life', '#motivation', '#inspiration', '#love', '#happy', '#instagood', '#photooftheday'],
    niche: ['#minimalism', '#productivity', '#selfimprovement', '#dailyroutine', '#lifegoals', '#positivity'],
    trending: ['#intentionalliving', '#slowliving', '#mindfullife', '#gratitude', '#manifestation']
  }
};

const DEFAULT_HASHTAGS = {
  popular: ['#content', '#creator', '#social', '#viral', '#trending', '#follow', '#share', '#explore'],
  niche: ['#contentcreator', '#creativecontent', '#digitalcreator', '#creatoreconomy', '#socialmediatips'],
  trending: ['#ai', '#2024', '#growth', '#authentic', '#community']
};

// Emojis
const EMOJIS = {
  smileys: ['😀', '😃', '😄', '😁', '😆', '😅', '🤣', '😂', '🙂', '😊', '😇', '🥰', '😍', '🤩', '😘', '😗', '😚', '😋', '😛', '😜', '🤪', '😝', '🤑', '🤗', '🤭', '🤫', '🤔', '🤐', '🤨', '😐', '😑', '😶', '😏', '😒', '🙄', '😬', '🤥', '😌', '😔', '😪', '🤤', '😴', '😷'],
  gestures: ['👍', '👎', '👊', '✊', '🤛', '🤜', '🤞', '✌️', '🤟', '🤘', '👌', '🤌', '🤏', '👈', '👉', '👆', '👇', '☝️', '✋', '🤚', '🖐️', '🖖', '👋', '🤙', '💪', '🙏', '👏', '🤝', '🙌', '👐', '🤲', '🫶'],
  symbols: ['✨', '⭐', '🌟', '💫', '⚡', '🔥', '💥', '❤️', '🧡', '💛', '💚', '💙', '💜', '🖤', '🤍', '💯', '✅', '❌', '⭕', '🎯', '💡', '🚀', '🎉', '🎊', '🏆', '🥇', '📈', '📉', '💰', '💎', '🔑', '🔗'],
  objects: ['💼', '📱', '💻', '🖥️', '⌨️', '🖱️', '📷', '🎬', '🎤', '🎧', '📚', '📝', '✏️', '📌', '📎', '🗂️', '📊', '📋', '📅', '🗓️', '⏰', '⌛', '🔔', '🔕', '📣', '💬', '💭', '🗨️', '📧', '📮', '📦', '🎁']
};

// State
let currentPlatform = 'twitter';
let selectedHashtags = [];
let drafts = [];
let customTemplates = [];
let theme = 'light';

// Initialize
document.addEventListener('DOMContentLoaded', async () => {
  await loadSettings();
  await loadDrafts();
  initTheme();
  initTabs();
  initCompose();
  initQuickTools();
  initTemplates();
  initHashtags();
  initCustomTemplates();
  initSaved();
  initSettings();
});

// ============================================
// SETTINGS & STORAGE
// ============================================

async function loadSettings() {
  try {
    const result = await chrome.storage.local.get(['theme', 'defaultPlatform', 'autoSave']);
    if (result.theme) theme = result.theme;
    if (result.defaultPlatform) currentPlatform = result.defaultPlatform;
  } catch (e) {
    console.log('Storage not available');
  }
}

async function loadDrafts() {
  try {
    const result = await chrome.storage.local.get(['drafts', 'customTemplates']);
    if (result.drafts) drafts = result.drafts;
    if (result.customTemplates) customTemplates = result.customTemplates;
  } catch (e) {
    console.log('Storage not available');
  }
}

async function saveCustomTemplates() {
  try {
    await chrome.storage.local.set({ customTemplates });
  } catch (e) {
    console.log('Storage not available');
  }
}

async function saveDrafts() {
  try {
    await chrome.storage.local.set({ drafts });
  } catch (e) {
    console.log('Storage not available');
  }
}

async function saveSettings() {
  try {
    await chrome.storage.local.set({ theme, defaultPlatform: currentPlatform });
  } catch (e) {
    console.log('Storage not available');
  }
}

// ============================================
// THEME
// ============================================

function initTheme() {
  const toggle = document.getElementById('themeToggle');
  
  // Apply saved theme
  if (theme === 'dark') {
    document.body.setAttribute('data-theme', 'dark');
  }
  
  toggle.addEventListener('click', () => {
    theme = theme === 'light' ? 'dark' : 'light';
    document.body.setAttribute('data-theme', theme);
    saveSettings();
  });
}

// ============================================
// TABS
// ============================================

function initTabs() {
  const tabs = document.querySelectorAll('.tab');
  const panels = document.querySelectorAll('.panel');
  
  tabs.forEach(tab => {
    tab.addEventListener('click', () => {
      const target = tab.dataset.tab;
      
      tabs.forEach(t => t.classList.remove('active'));
      panels.forEach(p => p.classList.remove('active'));
      
      tab.classList.add('active');
      document.getElementById(target).classList.add('active');
    });
  });
}

// ============================================
// COMPOSE
// ============================================

function initCompose() {
  const input = document.getElementById('contentInput');
  const platformBtns = document.querySelectorAll('.platform-btn');
  const charCount = document.getElementById('charCount');
  const wordCount = document.getElementById('wordCount');
  const remainingCount = document.getElementById('remainingCount');
  const progressFill = document.getElementById('progressFill');
  const progressPercent = document.getElementById('progressPercent');
  const copyBtn = document.getElementById('copyBtn');
  const clearBtn = document.getElementById('clearBtn');
  const saveDraftBtn = document.getElementById('saveDraftBtn');
  const emojiBtn = document.getElementById('emojiBtn');
  const emojiPicker = document.getElementById('emojiPicker');
  const closeEmoji = document.getElementById('closeEmoji');
  
  // Set initial platform
  platformBtns.forEach(btn => {
    if (btn.dataset.platform === currentPlatform) {
      btn.classList.add('active');
    }
  });
  
  // Platform selection
  platformBtns.forEach(btn => {
    btn.addEventListener('click', () => {
      platformBtns.forEach(b => b.classList.remove('active'));
      btn.classList.add('active');
      currentPlatform = btn.dataset.platform;
      updateCounter();
      saveSettings();
    });
  });
  
  // Update counter
  function updateCounter() {
    const text = input.value;
    const chars = text.length;
    const words = text.trim() ? text.trim().split(/\s+/).length : 0;
    const limit = PLATFORMS[currentPlatform].limit;
    const remaining = limit - chars;
    const percent = Math.min((chars / limit) * 100, 100);
    
    charCount.textContent = chars.toLocaleString();
    wordCount.textContent = words.toLocaleString();
    remainingCount.textContent = remaining.toLocaleString();
    progressPercent.textContent = Math.round(percent) + '%';
    progressFill.style.width = percent + '%';
    
    // Colors
    remainingCount.classList.remove('warning', 'danger');
    progressFill.classList.remove('warning', 'danger');
    
    if (percent > 100) {
      remainingCount.classList.add('danger');
      progressFill.classList.add('danger');
    } else if (percent > 80) {
      remainingCount.classList.add('warning');
      progressFill.classList.add('warning');
    }
  }
  
  input.addEventListener('input', updateCounter);
  updateCounter();
  
  // Copy
  copyBtn.addEventListener('click', async () => {
    if (!input.value.trim()) {
      showToast('Nothing to copy');
      return;
    }
    await navigator.clipboard.writeText(input.value);
    showToast('Copied to clipboard!');
  });
  
  // Clear
  clearBtn.addEventListener('click', () => {
    input.value = '';
    updateCounter();
  });
  
  // Save draft
  saveDraftBtn.addEventListener('click', () => {
    if (!input.value.trim()) {
      showToast('Nothing to save');
      return;
    }
    
    const draft = {
      id: Date.now().toString(),
      content: input.value,
      platform: currentPlatform,
      createdAt: new Date().toISOString()
    };
    
    drafts.unshift(draft);
    saveDrafts();
    renderDrafts();
    showToast('Draft saved!');
  });
  
  // Emoji picker
  emojiBtn.addEventListener('click', (e) => {
    e.stopPropagation();
    emojiPicker.classList.toggle('hidden');
    if (!emojiPicker.classList.contains('hidden')) {
      renderEmojis('smileys');
    }
  });
  
  closeEmoji.addEventListener('click', () => {
    emojiPicker.classList.add('hidden');
  });
  
  // Emoji tabs
  document.querySelectorAll('.emoji-tab').forEach(tab => {
    tab.addEventListener('click', () => {
      document.querySelectorAll('.emoji-tab').forEach(t => t.classList.remove('active'));
      tab.classList.add('active');
      renderEmojis(tab.dataset.category);
    });
  });
  
  // Close emoji picker when clicking outside
  document.addEventListener('click', (e) => {
    if (!emojiPicker.contains(e.target) && e.target !== emojiBtn) {
      emojiPicker.classList.add('hidden');
    }
  });
  
  function renderEmojis(category) {
    const grid = document.getElementById('emojiGrid');
    const emojis = EMOJIS[category] || EMOJIS.smileys;
    
    grid.innerHTML = emojis.map(emoji => 
      `<button class="emoji-item" data-emoji="${emoji}">${emoji}</button>`
    ).join('');
    
    grid.querySelectorAll('.emoji-item').forEach(item => {
      item.addEventListener('click', () => {
        const cursor = input.selectionStart;
        const before = input.value.substring(0, cursor);
        const after = input.value.substring(cursor);
        input.value = before + item.dataset.emoji + after;
        input.focus();
        input.selectionStart = input.selectionEnd = cursor + item.dataset.emoji.length;
        updateCounter();
      });
    });
  }
}

// ============================================
// QUICK TOOLS
// ============================================

function initQuickTools() {
  const input = document.getElementById('contentInput');
  
  // Uppercase
  document.getElementById('uppercaseBtn')?.addEventListener('click', () => {
    input.value = input.value.toUpperCase();
    input.dispatchEvent(new Event('input'));
    showToast('Converted to UPPERCASE');
  });
  
  // Lowercase
  document.getElementById('lowercaseBtn')?.addEventListener('click', () => {
    input.value = input.value.toLowerCase();
    input.dispatchEvent(new Event('input'));
    showToast('Converted to lowercase');
  });
  
  // Title Case
  document.getElementById('titlecaseBtn')?.addEventListener('click', () => {
    input.value = input.value.replace(/\w\S*/g, txt => 
      txt.charAt(0).toUpperCase() + txt.substr(1).toLowerCase()
    );
    input.dispatchEvent(new Event('input'));
    showToast('Converted to Title Case');
  });
  
  // Remove extra spaces
  document.getElementById('removeSpacesBtn')?.addEventListener('click', () => {
    input.value = input.value.replace(/\s+/g, ' ').trim();
    input.dispatchEvent(new Event('input'));
    showToast('Extra spaces removed');
  });
  
  // Add line breaks after sentences
  document.getElementById('addLineBreaksBtn')?.addEventListener('click', () => {
    input.value = input.value.replace(/([.!?])\s+/g, '$1\n\n');
    input.dispatchEvent(new Event('input'));
    showToast('Line breaks added');
  });
  
  // Remove emojis
  document.getElementById('removeEmojisBtn')?.addEventListener('click', () => {
    input.value = input.value.replace(/[\u{1F600}-\u{1F64F}]|[\u{1F300}-\u{1F5FF}]|[\u{1F680}-\u{1F6FF}]|[\u{1F1E0}-\u{1F1FF}]|[\u{2600}-\u{26FF}]|[\u{2700}-\u{27BF}]|[\u{1F900}-\u{1F9FF}]|[\u{1FA00}-\u{1FA6F}]|[\u{1FA70}-\u{1FAFF}]/gu, '');
    input.dispatchEvent(new Event('input'));
    showToast('Emojis removed');
  });
}

// ============================================
// TEMPLATES
// ============================================

function initTemplates() {
  const searchInput = document.getElementById('templateSearch');
  const chips = document.querySelectorAll('.category-chips .chip');
  const templateList = document.getElementById('templateList');
  
  let currentCategory = 'all';
  
  function renderTemplates(filter = '') {
    let templates = [];
    
    if (currentCategory === 'all') {
      Object.entries(TEMPLATES).forEach(([cat, items]) => {
        items.forEach(item => templates.push({ ...item, category: cat }));
      });
    } else {
      templates = (TEMPLATES[currentCategory] || []).map(item => ({ ...item, category: currentCategory }));
    }
    
    if (filter) {
      const f = filter.toLowerCase();
      templates = templates.filter(t => 
        t.title.toLowerCase().includes(f) || 
        t.content.toLowerCase().includes(f)
      );
    }
    
    templateList.innerHTML = templates.map(t => `
      <div class="template-card" data-content="${escapeHtml(t.content)}">
        <div class="template-card-header">
          <span class="template-title">${t.title}</span>
          <span class="template-category">${t.category}</span>
        </div>
        <p class="template-preview">${t.content.substring(0, 80)}...</p>
      </div>
    `).join('');
    
    templateList.querySelectorAll('.template-card').forEach(card => {
      card.addEventListener('click', async () => {
        const content = card.dataset.content;
        document.getElementById('contentInput').value = content;
        document.querySelector('.tab[data-tab="compose"]').click();
        document.getElementById('contentInput').dispatchEvent(new Event('input'));
        showToast('Template loaded!');
      });
    });
  }
  
  chips.forEach(chip => {
    chip.addEventListener('click', () => {
      chips.forEach(c => c.classList.remove('active'));
      chip.classList.add('active');
      currentCategory = chip.dataset.category;
      renderTemplates(searchInput.value);
    });
  });
  
  searchInput.addEventListener('input', () => {
    renderTemplates(searchInput.value);
  });
  
  renderTemplates();
}

// ============================================
// HASHTAGS
// ============================================

function initHashtags() {
  const input = document.getElementById('hashtagInput');
  const generateBtn = document.getElementById('generateHashtags');
  const typeTabs = document.querySelectorAll('.type-tab');
  const grid = document.getElementById('hashtagGrid');
  const selectedContainer = document.getElementById('selectedHashtags');
  const countEl = document.getElementById('hashtagCount');
  const copyBtn = document.getElementById('copyHashtags');
  
  let currentType = 'popular';
  let currentTopic = '';
  
  function renderHashtags() {
    const topicData = HASHTAGS[currentTopic.toLowerCase()] || DEFAULT_HASHTAGS;
    const hashtags = topicData[currentType] || DEFAULT_HASHTAGS[currentType];
    
    grid.innerHTML = hashtags.map(tag => `
      <button class="hashtag-tag ${selectedHashtags.includes(tag) ? 'selected' : ''}" data-tag="${tag}">${tag}</button>
    `).join('');
    
    grid.querySelectorAll('.hashtag-tag').forEach(tag => {
      tag.addEventListener('click', () => {
        const hashtag = tag.dataset.tag;
        if (selectedHashtags.includes(hashtag)) {
          selectedHashtags = selectedHashtags.filter(h => h !== hashtag);
          tag.classList.remove('selected');
        } else {
          selectedHashtags.push(hashtag);
          tag.classList.add('selected');
        }
        updateSelected();
      });
    });
  }
  
  function updateSelected() {
    selectedContainer.textContent = selectedHashtags.join(' ');
    countEl.textContent = selectedHashtags.length;
  }
  
  generateBtn.addEventListener('click', () => {
    currentTopic = input.value.trim() || 'general';
    renderHashtags();
    showToast('Hashtags generated!');
  });
  
  input.addEventListener('keypress', (e) => {
    if (e.key === 'Enter') generateBtn.click();
  });
  
  typeTabs.forEach(tab => {
    tab.addEventListener('click', () => {
      typeTabs.forEach(t => t.classList.remove('active'));
      tab.classList.add('active');
      currentType = tab.dataset.type;
      renderHashtags();
    });
  });
  
  copyBtn.addEventListener('click', async () => {
    if (selectedHashtags.length === 0) {
      showToast('Select some hashtags first');
      return;
    }
    await navigator.clipboard.writeText(selectedHashtags.join(' '));
    showToast('Hashtags copied!');
  });
  
  renderHashtags();
}

// ============================================
// CUSTOM TEMPLATES
// ============================================

function initCustomTemplates() {
  const addBtn = document.getElementById('addTemplateBtn');
  const modal = document.getElementById('addTemplateModal');
  const closeBtn = document.getElementById('closeAddTemplate');
  const saveBtn = document.getElementById('saveTemplateBtn');
  const nameInput = document.getElementById('templateName');
  const contentInput = document.getElementById('templateContent');
  
  addBtn?.addEventListener('click', () => {
    modal?.classList.remove('hidden');
    nameInput?.focus();
  });
  
  closeBtn?.addEventListener('click', () => {
    modal?.classList.add('hidden');
    nameInput.value = '';
    contentInput.value = '';
  });
  
  saveBtn?.addEventListener('click', () => {
    const name = nameInput?.value.trim();
    const content = contentInput?.value.trim();
    
    if (!name || !content) {
      showToast('Please fill both fields');
      return;
    }
    
    const template = {
      id: Date.now().toString(),
      title: name,
      content: content,
      createdAt: new Date().toISOString()
    };
    
    customTemplates.unshift(template);
    saveCustomTemplates();
    renderCustomTemplates();
    
    modal?.classList.add('hidden');
    nameInput.value = '';
    contentInput.value = '';
    showToast('Template saved!');
  });
  
  renderCustomTemplates();
}

function renderCustomTemplates() {
  const list = document.getElementById('customTemplateList');
  const empty = document.getElementById('emptyCustom');
  const modal = document.getElementById('addTemplateModal');
  
  if (!list) return;
  
  if (customTemplates.length === 0) {
    list.style.display = 'none';
    if (empty) empty.style.display = 'flex';
    return;
  }
  
  list.style.display = 'flex';
  if (empty) empty.style.display = 'none';
  
  list.innerHTML = customTemplates.map(t => `
    <div class="custom-card" data-id="${t.id}">
      <div class="custom-card-header">
        <span class="custom-card-title">${escapeHtml(t.title)}</span>
        <div class="custom-card-actions">
          <button class="use-custom" title="Use">
            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
              <path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"/>
              <path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"/>
            </svg>
          </button>
          <button class="copy-custom" title="Copy">
            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
              <rect x="9" y="9" width="13" height="13" rx="2" ry="2"/>
              <path d="M5 15H4a2 2 0 0 1-2-2V4a2 2 0 0 1 2-2h9a2 2 0 0 1 2 2v1"/>
            </svg>
          </button>
          <button class="delete-custom" title="Delete">
            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
              <polyline points="3 6 5 6 21 6"/>
              <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"/>
            </svg>
          </button>
        </div>
      </div>
      <p class="custom-card-preview">${escapeHtml(t.content)}</p>
    </div>
  `).join('');
  
  list.querySelectorAll('.custom-card').forEach(card => {
    const id = card.dataset.id;
    const template = customTemplates.find(t => t.id === id);
    
    card.querySelector('.use-custom')?.addEventListener('click', (e) => {
      e.stopPropagation();
      document.getElementById('contentInput').value = template.content;
      document.querySelector('.tab[data-tab="compose"]').click();
      document.getElementById('contentInput').dispatchEvent(new Event('input'));
      showToast('Template loaded!');
    });
    
    card.querySelector('.copy-custom')?.addEventListener('click', async (e) => {
      e.stopPropagation();
      await navigator.clipboard.writeText(template.content);
      showToast('Copied!');
    });
    
    card.querySelector('.delete-custom')?.addEventListener('click', (e) => {
      e.stopPropagation();
      customTemplates = customTemplates.filter(t => t.id !== id);
      saveCustomTemplates();
      renderCustomTemplates();
      showToast('Template deleted');
    });
  });
}

// ============================================
// SAVED DRAFTS
// ============================================

function initSaved() {
  const clearAll = document.getElementById('clearAllDrafts');
  
  clearAll.addEventListener('click', () => {
    if (drafts.length === 0) return;
    if (confirm('Clear all drafts?')) {
      drafts = [];
      saveDrafts();
      renderDrafts();
      showToast('All drafts cleared');
    }
  });
  
  renderDrafts();
}

function renderDrafts() {
  const list = document.getElementById('draftList');
  const empty = document.getElementById('emptyDrafts');
  
  if (drafts.length === 0) {
    list.style.display = 'none';
    empty.style.display = 'flex';
    return;
  }
  
  list.style.display = 'flex';
  empty.style.display = 'none';
  
  list.innerHTML = drafts.map(draft => `
    <div class="draft-card" data-id="${draft.id}">
      <div class="draft-meta">
        <div class="draft-platform">
          ${getPlatformIcon(draft.platform)}
        </div>
        <span class="draft-date">${formatDate(draft.createdAt)}</span>
      </div>
      <p class="draft-content">${escapeHtml(draft.content)}</p>
      <div class="draft-actions">
        <button class="btn btn-secondary use-btn">
          <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"/>
            <path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"/>
          </svg>
          Use
        </button>
        <button class="btn btn-secondary copy-btn">
          <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <rect x="9" y="9" width="13" height="13" rx="2" ry="2"/>
            <path d="M5 15H4a2 2 0 0 1-2-2V4a2 2 0 0 1 2-2h9a2 2 0 0 1 2 2v1"/>
          </svg>
          Copy
        </button>
        <button class="btn btn-secondary delete-btn">
          <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <polyline points="3 6 5 6 21 6"/>
            <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"/>
          </svg>
        </button>
      </div>
    </div>
  `).join('');
  
  list.querySelectorAll('.draft-card').forEach(card => {
    const id = card.dataset.id;
    const draft = drafts.find(d => d.id === id);
    
    card.querySelector('.use-btn').addEventListener('click', () => {
      document.getElementById('contentInput').value = draft.content;
      
      // Set platform
      document.querySelectorAll('.platform-btn').forEach(btn => {
        btn.classList.toggle('active', btn.dataset.platform === draft.platform);
      });
      currentPlatform = draft.platform;
      
      document.querySelector('.tab[data-tab="compose"]').click();
      document.getElementById('contentInput').dispatchEvent(new Event('input'));
    });
    
    card.querySelector('.copy-btn').addEventListener('click', async () => {
      await navigator.clipboard.writeText(draft.content);
      showToast('Copied!');
    });
    
    card.querySelector('.delete-btn').addEventListener('click', () => {
      drafts = drafts.filter(d => d.id !== id);
      saveDrafts();
      renderDrafts();
      showToast('Draft deleted');
    });
  });
}

function getPlatformIcon(platform) {
  const icons = {
    twitter: '<svg viewBox="0 0 24 24" fill="currentColor"><path d="M18.244 2.25h3.308l-7.227 8.26 8.502 11.24H16.17l-5.214-6.817L4.99 21.75H1.68l7.73-8.835L1.254 2.25H8.08l4.713 6.231zm-1.161 17.52h1.833L7.084 4.126H5.117z"/></svg>',
    linkedin: '<svg viewBox="0 0 24 24" fill="currentColor"><path d="M20.447 20.452h-3.554v-5.569c0-1.328-.027-3.037-1.852-3.037-1.853 0-2.136 1.445-2.136 2.939v5.667H9.351V9h3.414v1.561h.046c.477-.9 1.637-1.85 3.37-1.85 3.601 0 4.267 2.37 4.267 5.455v6.286zM5.337 7.433c-1.144 0-2.063-.926-2.063-2.065 0-1.138.92-2.063 2.063-2.063 1.14 0 2.064.925 2.064 2.063 0 1.139-.925 2.065-2.064 2.065zm1.782 13.019H3.555V9h3.564v11.452z"/></svg>',
    instagram: '<svg viewBox="0 0 24 24" fill="currentColor"><path d="M12 2.163c3.204 0 3.584.012 4.85.07 3.252.148 4.771 1.691 4.919 4.919.058 1.265.069 1.645.069 4.849 0 3.205-.012 3.584-.069 4.849-.149 3.225-1.664 4.771-4.919 4.919-1.266.058-1.644.07-4.85.07-3.204 0-3.584-.012-4.849-.07-3.26-.149-4.771-1.699-4.919-4.92-.058-1.265-.07-1.644-.07-4.849 0-3.204.013-3.583.07-4.849.149-3.227 1.664-4.771 4.919-4.919 1.266-.057 1.645-.069 4.849-.069zM12 0C8.741 0 8.333.014 7.053.072 2.695.272.273 2.69.073 7.052.014 8.333 0 8.741 0 12c0 3.259.014 3.668.072 4.948.2 4.358 2.618 6.78 6.98 6.98C8.333 23.986 8.741 24 12 24c3.259 0 3.668-.014 4.948-.072 4.354-.2 6.782-2.618 6.979-6.98.059-1.28.073-1.689.073-4.948 0-3.259-.014-3.667-.072-4.947-.196-4.354-2.617-6.78-6.979-6.98C15.668.014 15.259 0 12 0zm0 5.838a6.162 6.162 0 100 12.324 6.162 6.162 0 000-12.324zM12 16a4 4 0 110-8 4 4 0 010 8z"/></svg>',
    threads: '<svg viewBox="0 0 24 24" fill="currentColor"><path d="M12.186 24h-.007c-3.581-.024-6.334-1.205-8.184-3.509C2.35 18.44 1.5 15.586 1.472 12.01v-.017c.03-3.579.879-6.43 2.525-8.482C5.845 1.205 8.6.024 12.18 0h.014c2.746.02 5.043.725 6.826 2.098 1.677 1.29 2.858 3.13 3.509 5.467l-2.04.569c-1.104-3.96-3.898-5.984-8.304-6.015-2.91.022-5.11.936-6.54 2.717C4.307 6.504 3.616 8.914 3.589 12c.027 3.086.718 5.496 2.057 7.164 1.43 1.783 3.631 2.698 6.54 2.717 2.623-.02 4.358-.631 5.8-2.045 1.647-1.613 1.618-3.593 1.09-4.798-.31-.71-.873-1.3-1.634-1.75-.192 1.352-.622 2.446-1.284 3.272-.886 1.102-2.14 1.704-3.73 1.79-1.202.065-2.361-.218-3.259-.801-1.063-.689-1.685-1.74-1.752-2.96-.065-1.182.408-2.256 1.332-3.023.88-.73 2.132-1.13 3.634-1.166.96-.017 1.87.094 2.728.306-.084-.885-.351-1.574-.803-2.07-.535-.586-1.358-.9-2.45-.935l-.076-2.118c1.645.06 2.94.6 3.85 1.605.642.71 1.073 1.6 1.293 2.66.5.166.973.378 1.42.635 1.075.618 1.897 1.478 2.382 2.49.776 1.617.752 4.106-1.278 6.09-1.904 1.86-4.178 2.619-7.386 2.644z"/></svg>',
    facebook: '<svg viewBox="0 0 24 24" fill="currentColor"><path d="M24 12.073c0-6.627-5.373-12-12-12s-12 5.373-12 12c0 5.99 4.388 10.954 10.125 11.854v-8.385H7.078v-3.47h3.047V9.43c0-3.007 1.792-4.669 4.533-4.669 1.312 0 2.686.235 2.686.235v2.953H15.83c-1.491 0-1.956.925-1.956 1.874v2.25h3.328l-.532 3.47h-2.796v8.385C19.612 23.027 24 18.062 24 12.073z"/></svg>',
    tiktok: '<svg viewBox="0 0 24 24" fill="currentColor"><path d="M12.525.02c1.31-.02 2.61-.01 3.91-.02.08 1.53.63 3.09 1.75 4.17 1.12 1.11 2.7 1.62 4.24 1.79v4.03c-1.44-.05-2.89-.35-4.2-.97-.57-.26-1.1-.59-1.62-.93-.01 2.92.01 5.84-.02 8.75-.08 1.4-.54 2.79-1.35 3.94-1.31 1.92-3.58 3.17-5.91 3.21-1.43.08-2.86-.31-4.08-1.03-2.02-1.19-3.44-3.37-3.65-5.71-.02-.5-.03-1-.01-1.49.18-1.9 1.12-3.72 2.58-4.96 1.66-1.44 3.98-2.13 6.15-1.72.02 1.48-.04 2.96-.04 4.44-.99-.32-2.15-.23-3.02.37-.63.41-1.11 1.04-1.36 1.75-.21.51-.15 1.07-.14 1.61.24 1.64 1.82 3.02 3.5 2.87 1.12-.01 2.19-.66 2.77-1.61.19-.33.4-.67.41-1.06.1-1.79.06-3.57.07-5.36.01-4.03-.01-8.05.02-12.07z"/></svg>'
  };
  return icons[platform] || icons.twitter;
}

// ============================================
// SETTINGS
// ============================================

function initSettings() {
  const settingsBtn = document.getElementById('settingsBtn');
  const modal = document.getElementById('settingsModal');
  const closeBtn = document.getElementById('closeSettings');
  const backdrop = modal.querySelector('.modal-backdrop');
  const platformSelect = document.getElementById('defaultPlatformSelect');
  const autoSaveToggle = document.getElementById('autoSaveToggle');
  const clearAllBtn = document.getElementById('clearAllData');
  
  // Set initial values
  platformSelect.value = currentPlatform;
  
  settingsBtn.addEventListener('click', () => {
    modal.classList.remove('hidden');
  });
  
  const closeModal = () => modal.classList.add('hidden');
  closeBtn.addEventListener('click', closeModal);
  backdrop.addEventListener('click', closeModal);
  
  platformSelect.addEventListener('change', () => {
    currentPlatform = platformSelect.value;
    document.querySelectorAll('.platform-btn').forEach(btn => {
      btn.classList.toggle('active', btn.dataset.platform === currentPlatform);
    });
    saveSettings();
  });
  
  clearAllBtn.addEventListener('click', async () => {
    if (confirm('Clear all data including drafts and settings?')) {
      try {
        await chrome.storage.local.clear();
      } catch (e) {}
      drafts = [];
      selectedHashtags = [];
      renderDrafts();
      showToast('All data cleared');
      closeModal();
    }
  });
}

// ============================================
// UTILITIES
// ============================================

function escapeHtml(text) {
  const div = document.createElement('div');
  div.textContent = text;
  return div.innerHTML;
}

function formatDate(isoString) {
  const date = new Date(isoString);
  const now = new Date();
  const diff = now - date;
  const mins = Math.floor(diff / 60000);
  const hours = Math.floor(diff / 3600000);
  const days = Math.floor(diff / 86400000);
  
  if (mins < 1) return 'Just now';
  if (mins < 60) return `${mins}m ago`;
  if (hours < 24) return `${hours}h ago`;
  if (days < 7) return `${days}d ago`;
  return date.toLocaleDateString();
}

function showToast(message) {
  const toast = document.getElementById('toast');
  const msg = document.getElementById('toastMessage');
  
  msg.textContent = message;
  toast.classList.remove('hidden');
  
  setTimeout(() => {
    toast.classList.add('hidden');
  }, 2000);
}
