// ============================================
// SCRIPTRA BACKGROUND SERVICE WORKER
// ============================================

// Handle installation
chrome.runtime.onInstalled.addListener((details) => {
  if (details.reason === 'install') {
    // Set default settings
    chrome.storage.local.set({
      theme: 'light',
      defaultPlatform: 'twitter',
      autoSave: true,
      drafts: []
    });
    
    // Create context menus
    createContextMenus();
    
    console.log('[Scriptra] Extension installed');
  } else if (details.reason === 'update') {
    console.log('[Scriptra] Extension updated');
  }
});

// Create context menus
function createContextMenus() {
  chrome.contextMenus.create({
    id: 'scriptra-count',
    title: 'Count characters',
    contexts: ['selection']
  });
  
  chrome.contextMenus.create({
    id: 'scriptra-save',
    title: 'Save to Scriptra',
    contexts: ['selection']
  });
  
  chrome.contextMenus.create({
    id: 'scriptra-separator',
    type: 'separator',
    contexts: ['selection']
  });
  
  chrome.contextMenus.create({
    id: 'scriptra-open',
    title: 'Open Scriptra',
    contexts: ['selection']
  });
}

// Handle context menu clicks
chrome.contextMenus.onClicked.addListener(async (info, tab) => {
  const text = info.selectionText || '';
  
  switch (info.menuItemId) {
    case 'scriptra-count':
      const chars = text.length;
      const words = text.trim() ? text.trim().split(/\s+/).length : 0;
      
      // Show notification
      chrome.notifications.create({
        type: 'basic',
        iconUrl: 'icons/icon128.png',
        title: 'Character Count',
        message: `Characters: ${chars.toLocaleString()}\nWords: ${words.toLocaleString()}`,
        priority: 0
      });
      break;
      
    case 'scriptra-save':
      // Save to drafts
      const result = await chrome.storage.local.get(['drafts']);
      const drafts = result.drafts || [];
      
      const draft = {
        id: Date.now().toString(),
        content: text,
        platform: 'general',
        createdAt: new Date().toISOString()
      };
      
      drafts.unshift(draft);
      await chrome.storage.local.set({ drafts });
      
      chrome.notifications.create({
        type: 'basic',
        iconUrl: 'icons/icon128.png',
        title: 'Saved!',
        message: 'Text saved to Scriptra drafts',
        priority: 0
      });
      break;
      
    case 'scriptra-open':
      chrome.action.openPopup();
      break;
  }
});

// Handle messages from popup or content scripts
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  switch (message.type) {
    case 'GET_DRAFTS':
      chrome.storage.local.get(['drafts']).then(result => {
        sendResponse({ drafts: result.drafts || [] });
      });
      return true;
      
    case 'SAVE_DRAFT':
      chrome.storage.local.get(['drafts']).then(async result => {
        const drafts = result.drafts || [];
        drafts.unshift(message.draft);
        await chrome.storage.local.set({ drafts });
        sendResponse({ success: true });
      });
      return true;
      
    case 'GET_SETTINGS':
      chrome.storage.local.get(['theme', 'defaultPlatform', 'autoSave']).then(result => {
        sendResponse(result);
      });
      return true;
  }
});

// Handle keyboard shortcut
chrome.commands?.onCommand?.addListener((command) => {
  if (command === 'open-scriptra') {
    chrome.action.openPopup();
  }
});
