// ============================================
// SCRIPTRA CONTENT SCRIPT - PREMIUM VERSION
// ============================================

// Platform configuration
const PLATFORMS = {
  'twitter.com': { limit: 280, name: 'Twitter' },
  'x.com': { limit: 280, name: 'X' },
  'linkedin.com': { limit: 3000, name: 'LinkedIn' },
  'instagram.com': { limit: 2200, name: 'Instagram' },
  'facebook.com': { limit: 63206, name: 'Facebook' },
  'threads.net': { limit: 500, name: 'Threads' }
};

// Detect current platform
function detectPlatform() {
  const hostname = window.location.hostname.replace('www.', '');
  return PLATFORMS[hostname] || { limit: 280, name: 'Unknown' };
}

// Create counter element
function createCounter() {
  const counter = document.createElement('div');
  counter.className = 'scriptra-counter';
  counter.innerHTML = `
    <span class="scriptra-count">0</span>
    <span class="scriptra-divider">/</span>
    <span class="scriptra-limit">${detectPlatform().limit}</span>
  `;
  return counter;
}

// Update counter values
function updateCounter(counter, text, limit) {
  const count = text?.length || 0;
  const countEl = counter.querySelector('.scriptra-count');
  const percent = (count / limit) * 100;
  
  if (countEl) {
    countEl.textContent = count.toLocaleString();
  }
  
  // Update state classes
  counter.classList.remove('warning', 'danger');
  
  if (percent > 100) {
    counter.classList.add('danger');
  } else if (percent > 80) {
    counter.classList.add('warning');
  }
}

// Add counter to textarea
function addCounterToElement(element) {
  if (!element || element.dataset.scriptraEnhanced) return;
  
  const platform = detectPlatform();
  const counter = createCounter();
  
  // Make parent relative if needed
  const parent = element.parentElement;
  if (parent && getComputedStyle(parent).position === 'static') {
    parent.style.position = 'relative';
  }
  
  // Insert counter after element
  if (parent) {
    parent.style.position = 'relative';
    parent.appendChild(counter);
  }
  
  // Update on input
  const update = () => {
    const text = element.value || element.textContent || element.innerText || '';
    updateCounter(counter, text, platform.limit);
  };
  
  // Listen for changes
  element.addEventListener('input', update);
  element.addEventListener('keyup', update);
  
  // For contenteditable elements
  const observer = new MutationObserver(update);
  if (element.contentEditable === 'true') {
    observer.observe(element, { 
      childList: true, 
      subtree: true, 
      characterData: true 
    });
  }
  
  // Mark as enhanced
  element.dataset.scriptraEnhanced = 'true';
  
  // Initial update
  setTimeout(update, 100);
  
  return counter;
}

// Selectors for different platforms
const SELECTORS = {
  'twitter.com': [
    '[data-testid="tweetTextarea_0"]',
    '[data-testid="tweetTextarea_1"]',
    '[data-testid="dmComposerTextInput"]',
    '[role="textbox"][data-block="true"]'
  ],
  'x.com': [
    '[data-testid="tweetTextarea_0"]',
    '[data-testid="tweetTextarea_1"]',
    '[data-testid="dmComposerTextInput"]',
    '[role="textbox"][data-block="true"]'
  ],
  'linkedin.com': [
    '.ql-editor',
    '[contenteditable="true"][role="textbox"]',
    '.editor-content',
    '[data-placeholder="What do you want to talk about?"]'
  ],
  'instagram.com': [
    'textarea[placeholder*="caption"]',
    'textarea[placeholder*="comment"]',
    '[contenteditable="true"]'
  ],
  'facebook.com': [
    '[contenteditable="true"][role="textbox"]',
    '[data-lexical-editor="true"]',
    '.notranslate._5rpu'
  ],
  'threads.net': [
    '[contenteditable="true"]',
    'textarea'
  ]
};

// Find and enhance text areas
function enhanceTextAreas() {
  const hostname = window.location.hostname.replace('www.', '');
  const selectors = SELECTORS[hostname] || [];
  
  selectors.forEach(selector => {
    try {
      document.querySelectorAll(selector).forEach(element => {
        // Check if visible and not already enhanced
        if (element.offsetParent !== null && !element.dataset.scriptraEnhanced) {
          addCounterToElement(element);
        }
      });
    } catch (e) {
      // Ignore selector errors
    }
  });
}

// Observe DOM changes
const observer = new MutationObserver((mutations) => {
  let shouldCheck = false;
  
  for (const mutation of mutations) {
    if (mutation.addedNodes.length > 0) {
      shouldCheck = true;
      break;
    }
  }
  
  if (shouldCheck) {
    setTimeout(enhanceTextAreas, 300);
  }
});

// Initialize
function init() {
  // Start observing
  observer.observe(document.body, {
    childList: true,
    subtree: true
  });
  
  // Initial enhancement
  setTimeout(enhanceTextAreas, 500);
  
  // Periodic check for SPAs
  setInterval(enhanceTextAreas, 2000);
  
  console.log(`[Scriptra] Loaded for ${detectPlatform().name}`);
}

// Wait for DOM
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', init);
} else {
  init();
}
